/****************************************************************************
 *   $Id:: uart.c 7125 2011-04-15 00:22:12Z usb01267                        $
 *   Project: NXP LPC11U6x UART example
 *
 *   Description:
 *     This file contains UART code example which include UART 
 *     initialization, UART interrupt handler, and related APIs for 
 *     UART access.
 *
 ****************************************************************************
 * Software that is described herein is for illustrative purposes only
 * which provides customers with programming information regarding the
 * products. This software is supplied "AS IS" without any warranties.
 * NXP Semiconductors assumes no responsibility or liability for the
 * use of the software, conveys no license or title under any patent,
 * copyright, or mask work right to the product. NXP Semiconductors
 * reserves the right to make changes in the software without
 * notification. NXP Semiconductors also make no representation or
 * warranty that such application will be suitable for the specified
 * use without further testing or modification.
****************************************************************************/
#include "LPC11U6x.h"
#include "lpc11U6x_uart.h"

volatile uint32_t UARTStatus;
volatile uint8_t  UARTTxEmpty = 1;
volatile uint8_t  UARTBuffer[BUFSIZE];
volatile uint32_t UARTCount = 0;

/*****************************************************************************
** Function name:		USART_IRQHandler
**
** Descriptions:		USART interrupt handler
**
** parameters:			None
** Returned value:		None
** 
*****************************************************************************/
void USART_IRQHandler(void)
{
  uint8_t IIRValue, LSRValue;
  uint8_t Dummy = Dummy;

  IIRValue = LPC_USART0->IIR;
    
  IIRValue >>= 1;			/* skip pending bit in IIR */
  IIRValue &= 0x07;			/* check bit 1~3, interrupt identification */
  if (IIRValue == IIR_RLS)		/* Receive Line Status */
  {
    LSRValue = LPC_USART0->LSR;
    /* Receive Line Status */
    if (LSRValue & (LSR_OE | LSR_PE | LSR_FE | LSR_RXFE | LSR_BI))
    {
      /* There are errors or break interrupt */
      /* Read LSR will clear the interrupt */
      UARTStatus = LSRValue;
      Dummy = LPC_USART0->RBR;	/* Dummy read on RX to clear 
								interrupt, then bail out */
      return;
    }
    if (LSRValue & LSR_RDR)	/* Receive Data Ready */			
    {
      /* If no error on RLS, normal ready, save into the data buffer. */
      /* Note: read RBR will clear the interrupt */
      UARTBuffer[UARTCount++] = LPC_USART0->RBR;
      if (UARTCount == BUFSIZE)
      {
        UARTCount = 0;		/* buffer overflow */
      }	
    }
  }
  else if (IIRValue == IIR_RDA)	/* Receive Data Available */
  {
    /* Receive Data Available */
    UARTBuffer[UARTCount++] = LPC_USART0->RBR;
    if (UARTCount == BUFSIZE)
    {
      UARTCount = 0;		/* buffer overflow */
    }
  }
  else if (IIRValue == IIR_CTI)	/* Character timeout indicator */
  {
    /* Character Time-out indicator */
    UARTStatus |= 0x100;		/* Bit 9 as the CTI error */
  }
  else if (IIRValue == IIR_THRE)	/* THRE, transmit holding register empty */
  {
    /* THRE interrupt */
    LSRValue = LPC_USART0->LSR;		/* Check status in the LSR to see if
								valid data in U0THR or not */
    if (LSRValue & LSR_THRE)
    {
      UARTTxEmpty = 1;
    }
    else
    {
      UARTTxEmpty = 0;
    }
  }
  return;
}

#if MODEM_TEST
/*****************************************************************************
** Function name:		ModemInit
**
** Descriptions:		Initialize UART0 port as modem, setup pin select.
**
** parameters:			None
** Returned value:		None
** 
*****************************************************************************/
void ModemInit( void )
{
  
  LPC_IOCON->PIO0_7 &= ~0x07;     /* UART I/O config */
  LPC_IOCON->PIO0_7 |= 0x01;      /* UART CTS */
  LPC_IOCON->PIO0_17 &= ~0x07;    /* UART I/O config */
  LPC_IOCON->PIO0_17 |= 0x01;     /* UART RTS */
#if 1
  LPC_IOCON->PIO1_13 &= ~0x07;    /* UART I/O config */
  LPC_IOCON->PIO1_13 |= 0x01;     /* UART DTR */ 
  LPC_IOCON->PIO1_14 &= ~0x07;    /* UART I/O config */
  LPC_IOCON->PIO1_14 |= 0x01;     /* UART DSR */
  LPC_IOCON->PIO1_15 &= ~0x07;    /* UART I/O config */
  LPC_IOCON->PIO1_15 |= 0x01;     /* UART DCD */
  LPC_IOCON->PIO1_16 &= ~0x07;    /* UART I/O config */
  LPC_IOCON->PIO1_16 |= 0x01;     /* UART RI */

#else
  LPC_IOCON->PIO1_19 &= ~0x07;    /* UART I/O config */
  LPC_IOCON->PIO1_19 |= 0x01;     /* UART DTR */
  LPC_IOCON->PIO1_20 &= ~0x07;    /* UART I/O config */
  LPC_IOCON->PIO1_20 |= 0x01;     /* UART DSR */
  LPC_IOCON->PIO1_21 &= ~0x07;    /* UART I/O config */
  LPC_IOCON->PIO1_21 |= 0x01;     /* UART DCD */
  LPC_IOCON->PIO1_22 &= ~0x07;    /* UART I/O config */
  LPC_IOCON->PIO1_22 |= 0x01;     /* UART RI */
#endif
  LPC_USART0->MCR = 0xC0;          /* Enable Auto RTS and Auto CTS. */			
  return;
}
#endif

/*****************************************************************************
** Function name:		UARTInit
**
** Descriptions:		Initialize UART0 port, setup pin select,
**				clock, parity, stop bits, FIFO, etc.
**
** parameters:			UART baudrate
** Returned value:		None
** 
*****************************************************************************/
int UARTInit(uint32_t baudrate,int uclk)
{
  uint32_t Fdiv;
  uint32_t regVal;

  UARTTxEmpty = 1;
  UARTCount = 0;
  
  NVIC_DisableIRQ(USART_IRQn);
  /* Select only one location from below. */
#if 1
  LPC_IOCON->PIO0_18 &= ~0x07;    /*  UART I/O config */
  LPC_IOCON->PIO0_18 |= 0x01;     /* UART RXD */
  LPC_IOCON->PIO0_19 &= ~0x07;	
  LPC_IOCON->PIO0_19 |= 0x01;     /* UART TXD */
#endif
#if 0
  LPC_IOCON->PIO1_14 &= ~0x07;    /*  UART I/O config */
  LPC_IOCON->PIO1_14 |= 0x03;     /* UART RXD */
  LPC_IOCON->PIO1_13 &= ~0x07;	
  LPC_IOCON->PIO1_13 |= 0x03;     /* UART TXD */
#endif
#if 0
  LPC_IOCON->PIO1_17 &= ~0x07;    /*  UART I/O config */
  LPC_IOCON->PIO1_17 |= 0x02;     /* UART RXD */
  LPC_IOCON->PIO1_18 &= ~0x07;	
  LPC_IOCON->PIO1_18 |= 0x02;     /* UART TXD */
#endif
#if 0
  LPC_IOCON->PIO1_26 &= ~0x07;    /*  UART I/O config */
  LPC_IOCON->PIO1_26 |= 0x02;     /* UART RXD */
  LPC_IOCON->PIO1_27 &= ~0x07;	
  LPC_IOCON->PIO1_27 |= 0x02;     /* UART TXD */
#endif
  /* Enable UART clock */
  LPC_SYSCON->SYSAHBCLKCTRL |= (1<<12);
  LPC_SYSCON->USART0CLKDIV = 0x1;     /* divided by 1 */

  LPC_USART0->LCR = 0x83;             /* 8 bits, no Parity, 1 Stop bit */
  regVal = LPC_SYSCON->USART0CLKDIV;
  Fdiv = ((SystemCoreClock/regVal)/16)/baudrate ;	/*baud rate */

  LPC_USART0->DLM = Fdiv / 256;							
  LPC_USART0->DLL = Fdiv % 256;
  LPC_USART0->LCR = 0x03;		/* DLAB = 0 */
  LPC_USART0->FCR = 0x07;		/* Enable and reset TX and RX FIFO. */

  /* Read to clear the line status. */
  regVal = LPC_USART0->LSR;

  /* Ensure a clean start, no data in either TX or RX FIFO. */
  while (( LPC_USART0->LSR & (LSR_THRE|LSR_TEMT)) != (LSR_THRE|LSR_TEMT) );
  while ( LPC_USART0->LSR & LSR_RDR )
  {
	regVal = LPC_USART0->RBR;	/* Dump data from RX FIFO */
  }
 
  /* Enable the UART Interrupt */
  NVIC_EnableIRQ(USART_IRQn);

#if TX_INTERRUPT
  LPC_USART0->IER = IER_THRE | IER_RLS;	/* Enable UART interrupt */
#else
  LPC_USART0->IER = IER_RBR | IER_RLS;	/* Enable UART interrupt */
#endif
  return 0;
  
	
 
}

int UARTformat(int bits, int parity, int stop_bits )
{
  uint32_t value;
  value = LPC_USART0->LCR;
  value &= ~(USART_CFG_DATALEN(8) | USART_CFG_PARITY_EVEN | USART_CFG_PARITY_ENABLE | USART_CFG_STOPLEN(2));
	if (parity != 0) {
		//pCfg->regs->LCR = value | USART_CFG_DATALEN(bits) | (parity << USART_CFG_PARITY_EVEN_P) | USART_CFG_PARITY_ENABLE | USART_CFG_STOPLEN(stop_bits);
		value = value | USART_CFG_DATALEN(bits) | ( (3-parity) << USART_CFG_PARITY_EVEN_P) | USART_CFG_PARITY_ENABLE | USART_CFG_STOPLEN(stop_bits);
		LPC_USART0->LCR = value;
		} else {
		LPC_USART0->LCR = value | USART_CFG_DATALEN(bits) | USART_CFG_STOPLEN(stop_bits);		
	}
  return 0;
}
/*****************************************************************************
** Function name:		UARTSend
**
** Descriptions:		Send a block of data to the UART 0 port based
**				on the data length
**
** parameters:		buffer pointer, and data length
** Returned value:	None
** 
*****************************************************************************/
void UARTSend(uint8_t *BufferPtr)
{
  while ( *BufferPtr != '\0' )
  {
	  /* THRE status, contain valid data */
#if !TX_INTERRUPT
	  while ( !(LPC_USART0->LSR & LSR_THRE) );
	  LPC_USART0->THR = *BufferPtr;
#else
	  /* Below flag is set inside the interrupt handler when THRE occurs. */
      while ( !(UARTTxEmpty & 0x01) );
	  LPC_USART0->THR = *BufferPtr;
      UARTTxEmpty = 0;	/* not empty in the THR until it shifts out */
#endif
      BufferPtr++;
      //Length--;
  }
  return;
}

/*****************************************************************************
** Function name:		print_string
**
** Descriptions:		print out string on the terminal
**
** parameters:			pointer to the string end with NULL char.
** Returned value:		none.
** 
*****************************************************************************/
void print_string( uint8_t *str_ptr )
{
  while(*str_ptr != 0x00)
  {
    while((LPC_USART0->LSR & 0x60) != 0x60);
    LPC_USART0->THR = *str_ptr;
    str_ptr++;
  }
  return;
}

/*****************************************************************************
** Function name:		get_key
**
** Descriptions:		Get a character from the terminal
**
** parameters:			None
** Returned value:		character, zero is none.
** 
*****************************************************************************/
uint8_t get_key()
{
 uint8_t dummy;
  
  while ( !(LPC_USART0->LSR & 0x01) );  
  dummy = LPC_USART0->RBR;

 if ((dummy>='A') && (dummy<='Z'))
 {
	/* convert capital to non-capital character, A2a, B2b, C2c. */ 
	dummy +=('a'-'A');
 }
	
  return(dummy);
}

/******************************************************************************
**                            End Of File
******************************************************************************/
